define([
    'underscore',
    'jquery',
    'backbone',
    'marionette',

    'modules/new-appointment-request/views/common/scheduling-steps/heading-view',
    'modules/new-appointment-request/views/common/method/section-view',

    'modules/new-appointment-request/resources/common/pac-team/collection',
    'modules/new-appointment-request/resources/direct/clinics/collection',
    'modules/new-appointment-request/resources/common/recent-visits/model',
    'modules/new-appointment-request/resources/common/request-limit/model',

    'modules/appointments/appointments-module',
    'text!modules/new-appointment-request/views/common/scheduling-steps/templates/step-region.html',
], function(
    _,
    $,
    Backbone,
    Mn,

    HeadingView,
    SectionView,

    PactCollection,
    ClinicsCollection,
    RecentVisitModel,
    LimitModel,

    module,
    template
) {
    'use strict';


    var PAC_TEAM = 'pacTeam';
    var REQUEST_LIMIT = 'requestLimit';
    var CLINICS = 'clinics';


    var DIRECT_HISTORY = 'directHistory';
    var REQUEST_HISTORY = 'requestHistory';

    var PRIMARY_CARE_ID = '323';

    /**
     * @class MethodLayoutView
     * @typedef {Backbone.View<Backbone.Model>}
     */
    var View = Mn.View.extend({
        template: _.template(template),

        regions: {
            heading: '.step-heading',
            section: '.step-section',
        },

        // eslint-disable-next-line complexity
        initialize: function(options) {
            var isRequestSupported = options.isRequestSupported;
            var isDirectSupported = options.isDirectSchedulingSupported;
            var unfilteredClinics = this.model.get('unfilteredClinics');

            options.isLimitNeeded = isRequestSupported;
            options.isDirectHistoryNeeded = isDirectSupported && !this.model.isPrimaryCare();
            options.isRequestHistoryNeeded = isRequestSupported;

            this.filteredClinics = new ClinicsCollection();

            if (options.isPacNeeded) {
                this._getSuccessFunction(PAC_TEAM);
                this.listenTo(this.model.pacTeam(), 'fetch:success', this.setPacTeam);
            }

            if (isDirectSupported) {
                if (this.model.getCareId() === PRIMARY_CARE_ID && unfilteredClinics && unfilteredClinics.length) {
                    this._filterClinicList();
                } else {
                    this._fetchClinicList();
                }
            }

            if (options.isDirectHistoryNeeded) {
                this._fetchRecentHistory(true);
            }

            if (options.isRequestHistoryNeeded) {
                this._fetchRecentHistory(false);
            }

            if (options.isLimitNeeded) {
                this._fetchRequestLimit();
            }
        },

        onRender: function() {
            this.showHeading();
            this.showSectionView();
        },

        /**
         * Shows the title information for the view
         * @return {void}
         */
        showHeading: function() {
            var region = this.getRegion('heading');
            var model = new Backbone.Model({
                title: View.TITLE,
                number: View.STEP_NUMBER,
                instructions: View.INSTRUCTIONS,
            });
            region.show(new HeadingView({model: model}));
        },

        /**
         * Shows the body of the view
         * @return {void}
         */
        showSectionView: function() {
            var region;
            if (!this.isReadyToShow()) {
                return;
            }

            region = this.getRegion('section');
            region.show(new SectionView(this.options));
        },

        /**
         * Generator function to create fetch:success callbacks
         * @param {string} key
         * @return {function}
         * @private
         */
        _getSuccessFunction: function(key) {
            this.options[key] = null;
            return _.partial(this._onFetchSuccess, key);
        },

        setPacTeam: function() {
            var collection = this.model.pacTeam();
            var setPacTeam = _.bind(this._getSuccessFunction(PAC_TEAM), this);
            this.options.pacTeam = collection;

            return setPacTeam(collection);
        },

        _fetchClinicList: function() {
            var clinics = new ClinicsCollection();
            var institution = this.model.getInstitutionCode();
            var parentFacilityCode = this.model.getSiteCode();
            var typeOfCare = this.model.typeOfCare();
            var typeOfCareId = typeOfCare.get('id');
            var onClinicFetchSuccess = this._getSuccessFunction(CLINICS);

            this.listenTo(clinics, 'fetch:success', function () {
                onClinicFetchSuccess.call(this, clinics);
                this.triggerMethod('store:clinics', clinics);
            });

            clinics.fetchClinics(parentFacilityCode, institution, typeOfCareId);
        },

        _filterClinicList: function() {
            var clinics = this.model.get('unfilteredClinics');

            var institution = this.model.getInstitutionCode();

            this.options[CLINICS] = this.filteredClinics.reset(clinics.findWhere({ institutionCode: institution }));

            this.triggerMethod('store:clinics', this.filteredClinics);
        },

        /**
         * Retrieves the data required to determine the patient has received selected type
         * of care within the allotted time period to schedule request
         * @param {boolean} isDirectHistory
         * @return {void}
         * @private
         */
        _fetchRecentHistory: function(isDirectHistory) {
            var recent = new RecentVisitModel();

            var institution = this.model.getInstitutionCode();
            var facilityCode = this.model.getSiteCode();
            var typeOfCare = this.model.typeOfCare();
            var typeOfCareId = typeOfCare.get('id');
            var historyField;

            if (isDirectHistory) {
                historyField = DIRECT_HISTORY;
                recent.setAsDirectModel();
            } else {
                historyField = REQUEST_HISTORY;
                recent.setAsRequestModel();
            }

            this.listenTo(recent, 'fetch:success', this._getSuccessFunction(historyField));
            recent.fetchRecentVisits(facilityCode, institution, typeOfCareId);
        },

        /**
         * Retrieves the information required to determine if the patient is able to make
         * more requests at the selected institution
         * @return {void}
         * @private
         */
        _fetchRequestLimit: function() {
            var limit = new LimitModel();
            var institution = this.model.getInstitutionCode();
            var typeOfCare = this.model.typeOfCare();
            var typeOfCareId = typeOfCare.get('id');

            this.listenTo(limit, 'fetch:success', this._getSuccessFunction(REQUEST_LIMIT));
            limit.fetchRequestLimit(institution, typeOfCareId);
        },

        /**
         * Invoked after a network call returns in success
         * @param {string} key
         * @param {Backbone.Model|Backbone.Collection} model
         * @return {void}
         * @private
         */
        _onFetchSuccess: function(key, model) {
            this.options[key] = model;
            this.showSectionView();
        },

        /**
         * Checks if all pending network calls have returned and the view is rendered
         * @return {boolean}
         */
        // eslint-disable-next-line complexity
        isReadyToShow: function() {
            var options = this.options;
            var pactComplete = options.isPacNeeded ? !_.isNull(options[PAC_TEAM]) : true;
            var directHistoryComplete = options.isDirectHistoryNeeded ? !_.isNull(options[DIRECT_HISTORY]) : true;
            var requestHistoryComplete = options.isRequestHistoryNeeded ? !_.isNull(options[REQUEST_HISTORY]) : true;
            var requestLimitComplete = options.isLimitNeeded ? !_.isNull(options[REQUEST_LIMIT]) : true;
            var clinicsComplete = options.isDirectSchedulingSupported ? !_.isNull(options[CLINICS]) : true;

            return this.isRendered &&
                pactComplete &&
                directHistoryComplete &&
                requestHistoryComplete &&
                requestLimitComplete &&
                clinicsComplete;
        },
    });

    View.TITLE = 'Scheduling Method';
    View.STEP_NUMBER = 4;
    View.INSTRUCTIONS = 'The location you selected supports the following scheduling methods:';

    return View;
});
